/*
 * Decompiled with CFR 0.152.
 */
package net.shadew.json;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import net.shadew.json.CharUtil;
import net.shadew.json.FormattingConfig;
import net.shadew.json.JsonNode;
import net.shadew.json.JsonType;
import net.shadew.json.StringNode;

class Serializer {
    private static final ThreadLocal<Serializer> SERIALIZER_INSTANCE = ThreadLocal.withInitial(Serializer::new);
    private final Set<String> validIds = new HashSet<String>();
    private final Map<String, String> stringToJsonCache = new HashMap<String, String>();
    private final StringBuilder builder = new StringBuilder();
    private Appendable output;
    private FormattingConfig config;
    private int nextSpacing;
    private char quote;
    private int indent = 0;

    Serializer() {
    }

    static void serialize(JsonNode node, Appendable output, FormattingConfig config) throws IOException {
        Serializer serializer = SERIALIZER_INSTANCE.get();
        serializer.reset(output, config);
        serializer.writeJson(node);
    }

    void reset(Appendable output, FormattingConfig config) {
        this.builder.setLength(0);
        this.output = output;
        this.config = config;
        this.quote = (char)(config.useSingleQuoteStrings() && config.json5() ? 39 : 34);
        this.indent = 0;
        this.nextSpacing = 0;
    }

    private void addSpacing(int count) throws IOException {
        if (this.nextSpacing > count) {
            count = this.nextSpacing;
        }
        this.nextSpacing = 0;
        this.addSpaces(count);
    }

    private void addSpaces(int count) throws IOException {
        this.output.append(" ".repeat(count));
    }

    private void addTabs(int count) throws IOException {
        this.output.append("\t".repeat(count));
    }

    private void addNewline() throws IOException {
        this.output.append(this.config.lineSeparator().toString());
        this.addIndent(this.indent);
    }

    private void addIndent(int count) throws IOException {
        if (this.config.tabIndent()) {
            this.addTabs(this.config.indent() * count);
        } else {
            this.addSpaces(this.config.indent() * count);
        }
        this.nextSpacing = 0;
    }

    private String stringToJson(String str) {
        if (this.stringToJsonCache.containsKey(str)) {
            return this.stringToJsonCache.get(str);
        }
        this.builder.setLength(0);
        StringNode.quote(str, this.builder, this.quote);
        String converted = this.builder.toString();
        this.stringToJsonCache.put(str, converted);
        return converted;
    }

    private String keyToJson(String str) {
        if (this.config.json5() && this.config.useIdentifierKeys()) {
            if (this.validIds.contains(str)) {
                return str;
            }
            if (CharUtil.isIdentifierValid(str)) {
                this.validIds.add(str);
                return str;
            }
        }
        return this.stringToJson(str);
    }

    private int getObjectKeyAlignmentLength(JsonNode object) {
        if (!this.config.alignObjectValues()) {
            return 0;
        }
        int len = 0;
        for (String key : object.keySet()) {
            String jsonKey = this.keyToJson(key);
            len = Math.max(len, jsonKey.length());
        }
        return len;
    }

    private void writeString(String str) throws IOException {
        this.addSpacing(0);
        this.output.append(this.stringToJson(str));
    }

    private void writeNumber(BigDecimal decimal) throws IOException {
        this.addSpacing(0);
        String str = decimal.toString();
        if (this.config.ensurePointInNumbers()) {
            if (!str.contains(".") && !str.contains("e")) {
                this.output.append(str).append(".0");
            } else {
                this.output.append(str);
            }
        } else if (str.contains("e")) {
            this.output.append(str);
        } else {
            try {
                BigInteger integer = decimal.toBigIntegerExact();
                this.output.append(integer.toString());
            }
            catch (ArithmeticException exc) {
                this.output.append(decimal.toString());
            }
        }
    }

    private void writeBoolean(boolean bool) throws IOException {
        this.addSpacing(0);
        this.output.append(bool ? "true" : "false");
    }

    private void writeNull() throws IOException {
        this.addSpacing(0);
        this.output.append("null");
    }

    private void writeComma() throws IOException {
        this.addSpacing(this.config.spacesBeforeComma());
        this.output.append(",");
        this.nextSpacing = this.config.spacesAfterComma();
    }

    private void writeColon() throws IOException {
        this.addSpacing(this.config.spacesBeforeColon());
        this.output.append(":");
        this.nextSpacing = this.config.spacesAfterColon();
    }

    private void writeArray(JsonNode array) throws IOException {
        if (array.size() == 0) {
            this.addSpacing(this.config.spacesAroundArray());
            this.output.append('[');
            this.addSpacing(this.config.spacesWithinEmptyArray());
            this.output.append(']');
            this.nextSpacing = this.config.spacesAroundArray();
        } else {
            this.addSpacing(this.config.spacesAroundArray());
            this.output.append('[');
            this.nextSpacing = this.config.spacesWithinArray();
            boolean wrap = this.config.wrapArrays(array);
            if (wrap) {
                ++this.indent;
                this.addNewline();
            }
            int size = array.size();
            for (JsonNode node : array) {
                this.writeValue(node);
                if (size != 1 || this.config.json5() && this.config.addTrailingComma()) {
                    this.writeComma();
                }
                if (size != 1 && wrap) {
                    this.addNewline();
                }
                --size;
            }
            this.nextSpacing = Math.max(this.nextSpacing, this.config.spacesWithinArray());
            if (wrap) {
                --this.indent;
                this.addNewline();
            }
            this.addSpacing(0);
            this.output.append(']');
            this.nextSpacing = this.config.spacesAroundArray();
        }
    }

    private void writeObject(JsonNode object) throws IOException {
        if (object.size() == 0) {
            this.addSpacing(this.config.spacesAroundObject());
            this.output.append('{');
            this.addSpacing(this.config.spacesWithinEmptyObject());
            this.output.append('}');
            this.nextSpacing = this.config.spacesAroundObject();
        } else {
            this.addSpacing(this.config.spacesAroundObject());
            this.output.append('{');
            this.nextSpacing = this.config.spacesWithinObject();
            int alignmentLen = this.getObjectKeyAlignmentLength(object);
            boolean wrap = this.config.wrapArrays(object);
            if (wrap) {
                ++this.indent;
                this.addNewline();
            }
            int size = object.size();
            for (String key : object.keySet()) {
                String jsonKey = this.keyToJson(key);
                this.writeAlignedKey(jsonKey, alignmentLen);
                this.writeValue(object.get(key));
                if (size != 1 || this.config.json5() && this.config.addTrailingComma()) {
                    this.writeComma();
                }
                if (size != 1 && wrap) {
                    this.addNewline();
                }
                --size;
            }
            this.nextSpacing = Math.max(this.nextSpacing, this.config.spacesWithinObject());
            if (wrap) {
                --this.indent;
                this.addNewline();
            }
            this.addSpacing(0);
            this.output.append('}');
            this.nextSpacing = this.config.spacesAroundObject();
        }
    }

    private void writeAlignedKey(String key, int size) throws IOException {
        this.addSpacing(0);
        this.output.append(key);
        this.writeColon();
        if (!this.config.alignObjectValues()) {
            return;
        }
        int len = key.length();
        int remainingSpace = Math.max(size - len, 0);
        this.addSpaces(remainingSpace);
    }

    private void writeValue(JsonNode value) throws IOException {
        if (value.isNull()) {
            this.writeNull();
        } else if (value.isBoolean()) {
            this.writeBoolean(value.asBoolean());
        } else if (value.isNumber()) {
            this.writeNumber(value.asBigDecimal());
        } else if (value.isString()) {
            this.writeString(value.asExactString());
        } else if (value.isArray()) {
            this.writeArray(value);
        } else if (value.isObject()) {
            this.writeObject(value);
        } else assert (false);
    }

    private void writeJson(JsonNode node) throws IOException {
        if (this.config.makeNonExecutable()) {
            this.output.append(")]}'\n");
        }
        node.require(JsonType.ARRAY, JsonType.OBJECT);
        this.writeValue(node);
        if (this.config.newlineAtEnd()) {
            this.addNewline();
        }
        this.validIds.clear();
        this.stringToJsonCache.clear();
    }
}

